/* ------------------------------------------------------------------------
@NAME       : input.c
@DESCRIPTION: Routines for input of BibTeX data.
@GLOBALS    : LineNumber
@CALLS      : 
@CREATED    : 1997/10/14, Greg Ward (from code in bibparse.c)
@MODIFIED   : 
@VERSION    : $Id: input.c,v 1.3 1997/02/28 04:11:08 greg Exp $
-------------------------------------------------------------------------- */
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include "stdpccts.h"
#include "prototypes.h"
#include "error.h"

#define DEBUG 0

/* 
 * We maintain an independent line number counter here because zzline can
 * get out of sync with reality if the input is sufficiently garbled.  In
 * particular, certain syntax errors will cause the parser to bail out
 * before seeing the entire entry; thus, the zzline incrementing done by
 * the lexer won't be done for every newline.  The only place we're
 * guaranteed to see every newline is read_entry(), so we maintain
 * LineNumber there.  It's global so parse_entry() can use it to correct
 * zzline in case of syntax errors.
 */

static int   LineNumber;
char        *InputFilename;


/* ------------------------------------------------------------------------
@NAME       : bt_set_filename
@INPUT      : filename
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Sets the current input filename -- used for generating
              error and warning messages.
@GLOBALS    : InputFilename
@CALLS      : 
@CREATED    : Feb 1997, GPW
@MODIFIED   : 
-------------------------------------------------------------------------- */
void bt_set_filename (char *filename)
{
   InputFilename = filename;
}


/* ------------------------------------------------------------------------
@NAME       : read_entry
@INPUT      : infile      - file to read from
@OUTPUT     : *start_line
              *end_line   - set to the line number of the start/end of
                            the entry (actually, *end_line is set to
                            the start line of the next entry, which is
                            a bit undesirable... oh well)
@RETURNS    : string containing the entire unparsed text of one entry, or
              NULL if no '@' sign was seen on the input (ie. no more entries
              in file)
@DESCRIPTION: Reads from one '@' sign to the next, in order to isolate
              BibTeX entries for parsing.  '@' signs escaped with a 
              backslash don't count.
@GLOBALS    : 
@CALLS      : 
@CREATED    : Jan 1997, GPW
@MODIFIED   : 
-------------------------------------------------------------------------- */
static char
*read_entry (FILE *infile, int *start_line, int *end_line)
{
   static char *inbuf = NULL;
   static int   buf_size;
   int          pos = 0;
   int          escaped;
   int          c;

   if (inbuf == NULL)           /* no buffer -- must be starting new file */
   {
      buf_size = 2048;
      inbuf = (char *) malloc (buf_size * sizeof (char));
      LineNumber = 1;
   }

   /* First read and discard characters until we hit an unescaped @ */

   escaped = 0;
   while (!feof (infile))
   {
      c = fgetc (infile);
      if (c == '@' && !escaped) break;
      if (c == '\n') LineNumber++;
      escaped = (c == '\\');
   }

   /* 
    * Reached EOF without seeing an '@' -- we must be done, so free up  
    * the input buffer and reset inbuf to NULL -- that way, if we enter
    * read_entry() again we'll know that we're starting a new file.
    * We don't close the input file, though -- our caller opened, so
    * it should close it as well.
    */

   if (feof (infile))
   {
      free (inbuf);
      return (inbuf = NULL);
   }

   /* Otherwise, store the '@' that we did see, and then read and store */

   inbuf[pos++] = '@';
   *start_line = LineNumber;
   while (!feof (infile))
   {
      c = fgetc (infile);
      if (c == '@' && !escaped)
      {
         ungetc (c, infile);
         break;
      }
      if (c == '\n') LineNumber++;

      inbuf[pos++] = c;
      escaped = (c == '\\');

      if (pos == buf_size - 1)
      {
         buf_size *= 2;
         inbuf = realloc (inbuf, buf_size * sizeof (char));
      }
   }

   inbuf[pos] = 0;
   *end_line = LineNumber;
   return inbuf;
} /* read_entry() */


/* ------------------------------------------------------------------------
@NAME       : bt_parse_entry_s()
@INPUT      : entry_text - string containing the entire entry to parse,
                           or NULL meaning we're done, please cleanup
              options    - standard btparse options struct
              line       - current line number (if that makes any sense)
                           -- passed to the parser to set zzline, so that
                           lexical and syntax errors are properly localized
@OUTPUT     : *top       - newly-allocated AST for the entry
                           (or NULL if entry_text was NULL, ie. at EOF)
@RETURNS    : 1 with *top set to AST for entry on successful read/parse
              1 with *top==NULL if entry_text was NULL, ie. at EOF
              0 if any serious errors seen in input (*top is still 
                set to the AST, but only for as much of the input as we
                were able to parse)
              (A "serious" error is a lexical or syntax error; "trivial"
              errors such as warnings and notifications count as "success"
              for the purposes of this function's return value.)
@DESCRIPTION: Parses a BibTeX entry contained in a string.
@GLOBALS    : 
@CALLS      : ANTLR
@CREATED    : 1997/01/18, GPW (from code in bt_parse_entry())
@MODIFIED   : 
-------------------------------------------------------------------------- */
uchar bt_parse_entry_s (char         *entry_text,
                        bt_options_t *options,
                        int           line,
                        AST         **top)
{
   static int  *err_counts = NULL;
   uchar        ignore_emask;
   uchar        status;

#if DEBUG > 1
   printf ("bt_parse_entry_s: InputFilename=%s\n", InputFilename);
#endif

   *top = NULL;
   err_counts = get_error_counts (err_counts);

   if (entry_text == NULL)      /* signal to clean up */
   {
      free (err_counts);
      err_counts = NULL;
      return 1;
   }

   zzmode (START);
   zzast_sp = ZZAST_STACKSIZE;          /* pccts bug? */
   ANTLRs (entry (top, line), entry_text);
#if DEBUG
   dump_ast ("bt_parse_entry_s: single entry, after parsing:\n", *top);
#endif
   postprocess_entry (*top, options);
#if DEBUG
   dump_ast ("bt_parse_entry_s: single entry, after post-processing:\n", *top);
#endif


   /* 
    * This bit-twiddling fetches the error status (which has a bit
    * for each error class), masks off the bits for trivial errors
    * to get "true" if there were any serious errors, and then 
    * returns the opposite of that.
    */

   ignore_emask =
      (1<<E_NOTIFY) | (1<<E_CONTENT) | (1<<E_STRUCTURAL) | (1<<E_LEXWARN);
   status = !(error_status (err_counts) & ~ignore_emask);
   return status;

} /* bt_parse_entry_s () */


/* ------------------------------------------------------------------------
@NAME       : bt_parse_entry()
@INPUT      : infile  - file to read next entry from
              options - standard btparse options struct
@OUTPUT     : *top    - AST for the entry, or NULL if no entries left in file
@RETURNS    : same as bt_parse_entry_s()
@DESCRIPTION: Reads the text of a single entry from the given input file 
              (using read_entry()), and then calls bt_parse_entry_s()
              to parse that single entry.
@GLOBALS    : 
@CALLS      : bt_parse_entry_s()
@CREATED    : Jan 1997, GPW
@MODIFIED   : 
-------------------------------------------------------------------------- */
uchar bt_parse_entry (FILE *infile, bt_options_t *options, AST **top)
{
   char   *entry_buf;
   int     start_line, end_line;
   uchar   status;

   entry_buf = read_entry (infile, &start_line, &end_line);
   status = bt_parse_entry_s (entry_buf, options, start_line, top);
   return status;

} /* bt_parse_entry() */


/* ------------------------------------------------------------------------
@NAME       : bt_parse_file ()
@INPUT      : filename - name of file to open.  If NULL or "-", we read
                         from stdin rather than opening a new file.
              options
@OUTPUT     : top
@RETURNS    : 0 if any entries in the file had serious errors
              1 if all entries were OK
@DESCRIPTION: Parses an entire BibTeX file, and returns a linked list 
              of ASTs (or, if you like, a forest) for the entries in it.
              (Any entries with serious errors are omitted from the list.)
@GLOBALS    : 
@CALLS      : bt_parse_entry()
@CREATED    : 1997/01/18, from process_file() in bibparse.c
@MODIFIED   : 
@COMMENTS   : This function bears a *striking* resemblance to bibparse.c's
              process_file().  Eventually, I plan to replace this with 
              a generalized process_file() that takes a function pointer
              to call for each entry.  Until I decide on the right interface
              for that, though, I'm sticking with this simpler (but possibly
              memory-intensive) approach.
-------------------------------------------------------------------------- */
uchar bt_parse_file (char *filename, bt_options_t *options, AST **top)
{
   FILE   *infile;
   AST    *cur_entry, *last;
   uchar   status;

   /*
    * If a string was given, and it's *not* "-", then open that filename.
    * Otherwise just use stdin.
    */

   if (filename != NULL && strcmp (filename, "-") != 0)
   {
      InputFilename = filename;
      infile = fopen (filename, "r");
      if (infile == NULL)
      {
         perror (filename);
         return 0;
      }
   }
   else
   {
      InputFilename = "(stdin)";
      infile = stdin;
   }

   *top = NULL;
   last = NULL;
      
   status = 1;                  /* assume success */
   while (1)
   {
      status &= bt_parse_entry (infile, options, &cur_entry);
      if (!status) continue;            /* bad entry -- try next one */
      if (!cur_entry) break;            /* at eof -- we're done */
      if (last)
         last->right = cur_entry;
      else
         *top = cur_entry;

      last = cur_entry;
   }

   fclose (infile);
   InputFilename = NULL;
   return status;

} /* bt_parse_file() */
