/*
 * ff_filter.c
 *
 * Copyright (C) 2019 Peter Belkner <info@pbelkner.de>
 * Nanos gigantum humeris insidentes #TeamWhite
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2.0 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301  USA
 */
#include <libavutil/opt.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <ff.h>

#if 1 // {
#if defined (FF_BITMASK_BASED_CHANNEL_LAYOUT) // [
#undef FF_BITMASK_BASED_CHANNEL_LAYOUT
#endif // ]
#endif // }
#if defined (FF_FILTER_REDESIGNED) // [
#if defined (FF_HOLZHAMMER) // [
#undef FF_HOLZHAMMER
#endif // ]
#endif // }

#if defined (FF_HOLZHAMMER) // {
FF_DISABLE_DEPRECATION_WARNINGS // [
#endif // ]

///////////////////////////////////////////////////////////////////////////////
int ff_filter_create(ff_filter_t *f FFUNUSED,
    const AVCodecParameters *ocodecpar FFUNUSED,
    const AVCodecParameters *icodecpar FFUNUSED,
    AVRational time_base FFUNUSED, const char *descr)
{
  // cf. "<ffmpeg>/doc/examples/filtering_audio.c".
  int err=0;
#if defined (FF_FILTER_REDESIGNED) // [
  AVFilterGraph *filter_graph;
  AVFilterContext *abuffer_ctx;
  const AVFilter  *abuffer;
  AVFilterContext *volume_ctx;
  const AVFilter  *volume;
  AVFilterContext *aformat_ctx;
  const AVFilter  *aformat;
  AVFilterContext *abuffersink_ctx FFUNUSED;
  const AVFilter  *abuffersink FFUNUSED;

  //AVDictionary *options_dict;
  char options_str[1024];
  char ch_layout[64];

  ENTER(__func__);

  /* Create a new filtergraph, which will contain all the filters. */
  filter_graph = avfilter_graph_alloc();
  if (!filter_graph) {
    _DMESSAGE("Unable to create filter graph.\n");
    err= AVERROR(ENOMEM);
    goto e_filter_graph;
  }

  /* Create the abuffer filter;
   * it will be used for feeding the data into the graph. */
  abuffer = avfilter_get_by_name("abuffer");
  if (!abuffer) {
    _DMESSAGE("Could not find the abuffer filter.\n");
    err=AVERROR_FILTER_NOT_FOUND;
    goto e_abuffer;
  }

  abuffer_ctx = avfilter_graph_alloc_filter(filter_graph, abuffer, "src");
  if (!abuffer_ctx) {
    _DMESSAGE("Could not allocate the abuffer instance.\n");
    err= AVERROR(ENOMEM);
    goto e_abuffer_ctx;
  }

  av_channel_layout_describe(&icodecpar->ch_layout,
    ch_layout,sizeof(ch_layout));
#if 1 // [
  /* Set the filter options through the AVOptions API. */
  err=av_opt_set    (abuffer_ctx, "channel_layout", ch_layout,                            AV_OPT_SEARCH_CHILDREN);
  if (err<0) {
    _DMESSAGE("Could not set option 'channel_layout' to abuffer_ctx");
    goto e_abuffer_ctx_opt_set;
  }
  err=av_opt_set    (abuffer_ctx, "sample_fmt",     av_get_sample_fmt_name(icodecpar->format), AV_OPT_SEARCH_CHILDREN);
  if (err<0) {
    _DMESSAGE("Could not set option 'sample_fmt' to abuffer_ctx");
    goto e_abuffer_ctx_opt_set;
  }
#if 0 // [
  av_opt_set_q  (abuffer_ctx, "time_base",
      (AVRational){ 1, icodecpar->sample_rate },  AV_OPT_SEARCH_CHILDREN);
#else // ] [
  err=av_opt_set_q  (abuffer_ctx, "time_base",
      time_base,  AV_OPT_SEARCH_CHILDREN);
  if (err<0) {
    _DMESSAGE("Could not set option 'time_base' to abuffer_ctx");
    goto e_abuffer_ctx_opt_set;
  }
#endif // ]
  err=av_opt_set_int(abuffer_ctx, "sample_rate",    icodecpar->sample_rate,                     AV_OPT_SEARCH_CHILDREN);
  if (err<0) {
    _DMESSAGE("Could not set option 'sample_rate' to abuffer_ctx");
    goto e_abuffer_ctx_opt_set;
  }

  /* Now initialize the filter; we pass NULL options, since we have already
   * set all the options above. */
  err = avfilter_init_str(abuffer_ctx, NULL);
  if (err < 0) {
    _DMESSAGEV("Could not initialize the abuffer filter: %s.\n",
	av_err2str(err));
    goto e_abuffer_ctx_init;
  }
#else // ] [
  /* A third way of passing the options is in a string of the form
   * key1=value1:key2=value2.... */
  snprintf(options_str, sizeof(options_str),
      "sample_fmts=%s:sample_rates=%d:channel_layouts=%s",
      av_get_sample_fmt_name(icodecpar->format),
      icodecpar->sample_rate,
      ch_layout);
  err=avfilter_init_str(abuffer_ctx, options_str);
  if (err < 0) {
    _DMESSAGE("Could not initialize the abuffer filter.\n");
    fprintf(stderr,"%s\n",options_str);
    goto e_abuffer_ctx_init;
  }
#endif // ]

  /* Create volume filter. */
  volume=avfilter_get_by_name("volume");
  if (!volume) {
    _DMESSAGE("Could not find the volume filter.\n");
    err=AVERROR_FILTER_NOT_FOUND;
    goto e_volume;
  }

  volume_ctx=avfilter_graph_alloc_filter(filter_graph, volume, "volume");
  if (!volume_ctx) {
    _DMESSAGE("Could not allocate the volume instance.\n");
    err=AVERROR(ENOMEM);
    goto e_volume_ctx;
  }

#if 0 // [
  /* A different way of passing the options is as key/value pairs in a
   * dictionary. */
  PUTS(__FILE__,__LINE__,"\n");
  err=av_dict_set(&options_dict, "volume", descr, 0);
  PUTS(__FILE__,__LINE__,"\n");
  if (err<0) {
    _DMESSAGE("Could not set dictionary.\n");
    goto e_dict_set;
  }
  err = avfilter_init_dict(volume_ctx, &options_dict);
  av_dict_free(&options_dict);
  if (err < 0) {
    _DMESSAGE("Could not initialize the volume filter.\n");
    goto e_volume_ctx_init;
  }
#else // ] [
  err = avfilter_init_str(volume_ctx, descr);
  if (err < 0) {
    _DMESSAGE("Could not initialize the aformat filter.\n");
    goto e_volume_ctx_init;
  }
#endif // ]

  /* Create the aformat filter;
   * it ensures that the output is of the format we want. */
  aformat = avfilter_get_by_name("aformat");
  if (!aformat) {
    _DMESSAGE("Could not find the aformat filter.\n");
    err= AVERROR_FILTER_NOT_FOUND;
    goto e_aformat;
  }

  aformat_ctx = avfilter_graph_alloc_filter(filter_graph, aformat, "aformat");
  if (!aformat_ctx) {
    _DMESSAGE("Could not allocate the aformat instance.\n");
    err= AVERROR(ENOMEM);
    goto e_aformat_ctx;
  }

  /* A third way of passing the options is in a string of the form
   * key1=value1:key2=value2.... */
  snprintf(options_str, sizeof(options_str),
           "sample_fmts=%s:sample_rates=%d:channel_layouts=stereo",
           av_get_sample_fmt_name(AV_SAMPLE_FMT_S16), 44100);
  err = avfilter_init_str(aformat_ctx, options_str);
  if (err < 0) {
    _DMESSAGE("Could not initialize the aformat filter.\n");
    goto e_aformat_ctx_init;
  }

  /* Finally create the abuffersink filter;
   * it will be used to get the filtered data out of the graph. */
  abuffersink = avfilter_get_by_name("abuffersink");
  if (!abuffersink) {
    _DMESSAGE("Could not find the abuffersink filter.\n");
    err= AVERROR_FILTER_NOT_FOUND;
    goto e_abuffersink;
  }

  abuffersink_ctx = avfilter_graph_alloc_filter(filter_graph, abuffersink, "sink");
  if (!abuffersink_ctx) {
    _DMESSAGE("Could not allocate the abuffersink instance.\n");
    err= AVERROR(ENOMEM);
    goto e_abuffersink_ctx;
  }

  /* This filter takes no options. */
  err = avfilter_init_str(abuffersink_ctx, NULL);
  if (err < 0) {
    _DMESSAGEV("Could not initialize the abuffersink instance: %s.\n",
        av_err2str(err));
    goto e_abuffersink_ctx_init;
  }

  /* Connect the filters;
   * in this simple case the filters just form a linear chain. */
  err = avfilter_link(abuffer_ctx, 0, volume_ctx, 0);
  if (err >= 0)
    err = avfilter_link(volume_ctx, 0, aformat_ctx, 0);
  if (err >= 0)
    err = avfilter_link(aformat_ctx, 0, abuffersink_ctx, 0);
  if (err < 0) {
    _DMESSAGEV("Error connecting filters: %s\n",av_err2str(err));
    goto e_connect;
  }

  /* Configure the graph. */
  err = avfilter_graph_config(filter_graph,NULL);
  if (err < 0) {
    _DMESSAGEV("Error configuring the filter graph: %s\n",
	av_err2str(err));
    goto e_avfilter_graph_config;
  }

  f->graph=filter_graph;
  f->ctx.src=abuffer_ctx;
  f->ctx.sink=abuffersink_ctx;

  //return 0;
  return RETURN_INT(__FILE__,__LINE__,0);
e_avfilter_graph_config:
e_connect:
e_abuffersink_ctx_init:
  avfilter_free(abuffersink_ctx);
e_abuffersink_ctx:
e_abuffersink:
e_aformat_ctx_init:
  avfilter_free(aformat_ctx);
e_aformat_ctx:
e_aformat:
e_volume_ctx_init:
  avfilter_free(volume_ctx);
//e_dict_set:
e_volume_ctx:
e_volume:
e_abuffer_ctx_opt_set:
e_abuffer_ctx_init:
  avfilter_free(abuffer_ctx);
e_abuffer_ctx:
e_abuffer:
  avfilter_graph_free (&filter_graph);
e_filter_graph:
#else // ] [
#if defined (FF_SAMPLE_FMTS_DEPRECATED_FIX) // [
  const enum AVSampleFormat *sink_sample_fmts FFUNUSED;
#else // ] [
  const enum AVSampleFormat sink_sample_fmts[] FFUNUSED
      ={ ocodecpar->format,-1 };
#endif // ]
#if defined(FF_BITMASK_BASED_CHANNEL_LAYOUT) // [
  const int64_t sink_channel_layouts[] FFUNUSED
      ={ ocodecpar->channel_layout,-1 };
#endif // ]
#if defined (FF_SAMPLE_FMTS_DEPRECATED_FIX) // [
  const int *sink_sample_rates FFUNUSED;
#else // ] [
  const int sink_sample_rates[] FFUNUSED
      ={ ocodecpar->sample_rate,-1 };
#endif // ]
  char args[512];
  FF_CONST AVCodec *ocodec;

  struct {
    const AVFilter *f;
    AVFilterInOut *in;
  } src;

  struct {
    const AVFilter *f;
    AVFilterInOut *out;
  } sink;

  ENTER(__func__);

  /////////////////////////////////////////////////////////////////////////////
  f->graph=avfilter_graph_alloc();

  if (!f->graph) {
    _DMESSAGE("allocating filter graph");
    err=AVERROR(ENOMEM);
    goto e_graph;
  }

  /////////////////////////////////////////////////////////////////////////////
  // buffer audio source: the decoded frames from the decoder will
  // be inserted here.
  src.f=avfilter_get_by_name("abuffer");

  if (!src.f) {
    _DMESSAGE("audio filter \"abuffer\" not available");
    err=-1;
    goto e_src;
  }

  /////////////////////////////////////////////////////////////////////////////
#if defined (FF_BITMASK_BASED_CHANNEL_LAYOUT) // [
  snprintf(args,sizeof args,
      "time_base=%d/%d"
      ":sample_rate=%d"
      ":sample_fmt=%s"
      ":channel_layout=0x%"PRIx64,
      time_base.num,
      time_base.den,
      icodecpar->sample_rate,
      av_get_sample_fmt_name(icodecpar->format),
      icodecpar->channel_layout
			);
  err=avfilter_graph_create_filter(&f->ctx.src,src.f,"in",args,NULL,f->graph);

  if (err<0) {
    _DMESSAGEV("creating filter source: %s (%d)",av_err2str(err),err);
    goto e_srcctx;
  }
#else // ] [
  err=snprintf(args,sizeof args,
      "time_base=%d/%d"
      ":sample_rate=%d"
      ":sample_fmt=%s"
      ":channel_layout=",
      time_base.num,
      time_base.den,
      icodecpar->sample_rate,
      av_get_sample_fmt_name(icodecpar->format)
			);
  av_channel_layout_describe(&icodecpar->ch_layout,
    args+err,sizeof(args)-err);
//_DWARNINGV("\"%s\"",args);
  err=avfilter_graph_create_filter(&f->ctx.src,src.f,"in",args,NULL,f->graph);

  if (err<0) {
    _DMESSAGEV("creating filter source: %s (%d)",av_err2str(err),err);
    goto e_srcctx;
  }
#endif // ]

  /////////////////////////////////////////////////////////////////////////////
  // buffer audio sink: to terminate the filter chain.
  sink.f=avfilter_get_by_name("abuffersink");

  if (!sink.f) {
    _DMESSAGE("audio filter \"abuffersink\" not available");
    err=-1;
    goto e_sink;
  }

  /////////////////////////////////////////////////////////////////////////////
  err=avfilter_graph_create_filter(&f->ctx.sink,sink.f,"out",NULL,NULL,
      f->graph);

  if (err<0) {
    _DMESSAGEV("creating filter sink: %s (%d)",av_err2str(err),err);
    goto e_sinkctx;
  }

  /////////////////////////////////////////////////////////////////////////////
#if defined (FF_SAMPLE_FMTS_DEPRECATED_FIX) // [
  /*
  err= avcodec_get_supported_config(dec_ctx,NULL,AV_CODEC_CONFIG_PIX_FORMAT,
      0,(const void**)&sink_sample_fmts,NULL);
  */
  err=av_opt_set_bin(f->ctx.sink,"sample_fmts",
      sink_sample_fmts,-1,AV_OPT_SEARCH_CHILDREN|AV_OPT_TYPE_FLAG_ARRAY);
#else // ] [
  err=av_opt_set_int_list(f->ctx.sink,"sample_fmts",
      sink_sample_fmts,-1,AV_OPT_SEARCH_CHILDREN);
#endif // ]

  if (err<0) {
    _DMESSAGEV("setting output sample format: %s (%d)",av_err2str(err),err);
    goto e_sinkargs;
  }

  /////////////////////////////////////////////////////////////////////////////
#if defined (FF_BITMASK_BASED_CHANNEL_LAYOUT) // [
  err=av_opt_set_int_list(f->ctx.sink,"channel_layouts",
      sink_channel_layouts,-1,AV_OPT_SEARCH_CHILDREN);
#elif 1 // ] [
  av_channel_layout_describe(&ocodecpar->ch_layout,
    args,sizeof(args));
  err=av_opt_set(f->ctx.sink,"ch_layouts",args,AV_OPT_SEARCH_CHILDREN);
#endif // ]

  if (err<0) {
    _DMESSAGEV("setting output channel layout: %s (%d)",av_err2str(err),err);
    goto e_sinkargs;
  }

  /////////////////////////////////////////////////////////////////////////////
#if defined (FF_SAMPLE_FMTS_DEPRECATED_FIX) // [
#else // ] [
  err=av_opt_set_int_list(f->ctx.sink,"sample_rates",
      sink_sample_rates,-1,AV_OPT_SEARCH_CHILDREN);
#endif // ]

  if (err<0) {
    _DMESSAGEV("setting output sample rate: %s (%d)",av_err2str(err),err);
    goto e_sinkargs;
  }

  /////////////////////////////////////////////////////////////////////////////
  /*
   * Set the endpoints for the filter graph. The filter_graph will
   * be linked to the graph described by filters_descr.
   */

  /*
   * The buffer source output must be connected to the input pad of
   * the first filter described by filters_descr; since the first
   * filter input label is not specified, it is set to "in" by
   * default.
   */
  sink.out=avfilter_inout_alloc();

  if (!sink.out) {
    _DMESSAGE("allocating outputs");
    goto e_outputs;
  }

  sink.out->name=av_strdup("in");

  if (!sink.out->name) {
    _DMESSAGE("duplicating sink name");
    goto e_sinkname;
  }

  sink.out->filter_ctx=f->ctx.src;
  sink.out->pad_idx=0;
  sink.out->next= NULL;

  /*
   * The buffer sink input must be connected to the output pad of
   * the last filter described by filters_descr; since the last
   * filter output label is not specified, it is set to "out" by
   * default.
   */
  src.in=avfilter_inout_alloc();

  if (!src.in) {
    _DMESSAGE("allocating inputs");
    goto e_inputs;
  }

  src.in->name=av_strdup("out");

  if (!src.in->name) {
    _DMESSAGE("duplicating source name");
    goto e_srcname;
  }

  src.in->filter_ctx=f->ctx.sink;
  src.in->pad_idx=0;
  src.in->next=NULL;

  /////////////////////////////////////////////////////////////////////////////
  err=avfilter_graph_parse_ptr(f->graph,descr,&src.in,
      &sink.out,NULL);

  if (err<0) {
    _DMESSAGEV("parsing: %s (%d)",av_err2str(err),err);
    goto e_parse;
  }

  /////////////////////////////////////////////////////////////////////////////
  err=avfilter_graph_config(f->graph,NULL);

  if (err<0) {
    _DMESSAGEV("onfiguring: %s (%d)",av_err2str(err),err);
    goto e_config;
  }

  /////////////////////////////////////////////////////////////////////////////
  // needs to be called when the graph already has been linked.
  ocodec=avcodec_find_encoder(ocodecpar->codec_id);

  if (!ocodec) {
    _DMESSAGE("target codec doesn't exist");
    goto e_ocodec;
  }

  if (!(ocodec->capabilities & AV_CODEC_CAP_VARIABLE_FRAME_SIZE))
    av_buffersink_set_frame_size(f->ctx.sink,ocodecpar->frame_size);

  /////////////////////////////////////////////////////////////////////////////
  avfilter_inout_free(&src.in);
  avfilter_inout_free(&sink.out);

  ///////////////////////////////////////////////////////////////////////////
  //return 0;
  return RETURN_INT(__FILE__,__LINE__,0);
e_ocodec:
e_config:
e_parse:
e_srcname:
  avfilter_inout_free(&src.in);
e_inputs:
e_sinkname:
  avfilter_inout_free(&sink.out);
e_outputs:
e_sinkargs:
e_sinkctx:
e_sink:
e_srcctx:
e_src:
  avfilter_graph_free(&f->graph);
e_graph:
#endif // ]
  //return err;
  return RETURN_INT(__FILE__,__LINE__,err);
}

void ff_filter_destroy(ff_filter_t *f)
{
  avfilter_graph_free(&f->graph);
}

///////////////////////////////////////////////////////////////////////////////
int ff_filter_send_frame(ff_filter_t *f, AVFrame *frame)
{
  return av_buffersrc_add_frame(f->ctx.src,frame);
}

int ff_filter_receive_frame(ff_filter_t *f, AVFrame *frame)
{
  return av_buffersink_get_frame(f->ctx.sink,frame);
}

#if defined (FF_HOLZHAMMER) // {
FF_ENABLE_DEPRECATION_WARNINGS // ]
#endif // ]
